implementation module ospicture

//	Version 1.0

//
//	Drawing functions and other operations on Pictures. 
//

import clCrossCall
import osfont, ostoolbox
import commondef, StdPictureDef

::	Picture
	=	{	pictContext		:: !OSPictContext	// The context for drawing operations
		,	pictToolbox		:: !.OSToolbox		// The continuation value
		,	pictOrigin		:: !Origin			// The current origin of the picture
		,	pictPen			:: !Pen				// The current state of the pen
		}
::	Origin
	:==	Point
::	OSPictContext
	:==	HDC
::  Pen
	=	{	penSize			:: !Int				// The width and height of the pen
  		,	penForeColour	:: !Colour			// The drawing colour of the pen
		,	penBackColour	:: !Colour			// The background colour of the pen
		,	penPos			:: !Point			// The pen position in local coordinates
		,	penFont			:: !Font			// The font information to draw text and characters
		}


//	Conversion operations to and from Picture
/*
initialisePicture :: !Origin !Pen !OSPictContext !*OSToolbox -> (!OSPictContext,!*OSToolbox)
initialisePicture origin pen=:{penSize,penForeColour,penBackColour,penPos,penFont} hdc tb
	# {osfontname,osfontstyles,osfontsize}	= OSfontgetimp penFont
	# (hdc,tb)								= WinInitPicture
												penSize
												iModeCopy
												initforecolour
												initbackcolour
												initpen
												(osfontname,osfontstyles,osfontsize)
												(0,0)
												(hdc,tb)
	# (_,_,_,_,_,_,(hdc,tb))	= WinDonePicture (hdc,tb)
	= (hdc,tb)
where
	initforecolour	= toRGBtriple penForeColour
	initbackcolour	= toRGBtriple penBackColour
	initpen			= PointToTuple (penPos-origin)
*/
packPicture :: !Origin !Pen !OSPictContext !*OSToolbox -> *Picture
packPicture origin pen=:{penSize,penForeColour,penBackColour,penPos,penFont} hdc tb
	# {osfontname,osfontstyles,osfontsize}= OSfontgetimp penFont
	# (hdc,tb)		= WinInitPicture
						penSize
						iModeCopy
						initforecolour
						initbackcolour
						initpen
						(osfontname,osfontstyles,osfontsize)
						(0,0)
						(hdc,tb)
	= {	pictContext	= hdc
	  ,	pictToolbox	= tb
	  ,	pictOrigin	= origin
	  ,	pictPen		= pen
	  }
where
	initforecolour	= toRGBtriple penForeColour
	initbackcolour	= toRGBtriple penBackColour
	initpen			= PointToTuple (penPos-origin)

unpackPicture :: !*Picture -> (!Origin,!Pen,!OSPictContext,!*OSToolbox)
unpackPicture {pictOrigin,pictPen,pictContext,pictToolbox}
	# (_,_,_,_,_,_,(hdc,tb))	= WinDonePicture (pictContext,pictToolbox)
	= (pictOrigin,pictPen,hdc,tb)

peekPicture :: !*Picture -> (!Origin,!Pen,!OSPictContext,!*OSToolbox)
peekPicture {pictOrigin,pictPen,pictContext,pictToolbox}
	= (pictOrigin,pictPen,pictContext,pictToolbox)

unpeekPicture :: !Origin !Pen !OSPictContext !*OSToolbox -> *Picture
unpeekPicture origin pen hdc tb
	= {pictOrigin=origin,pictPen=pen,pictContext=hdc,pictToolbox=tb}

peekOSPictContext :: !*Picture -> (!OSPictContext,!*Picture)
peekOSPictContext picture=:{pictContext}
	= (pictContext,picture)

sharePicture :: !*Picture -> (!Picture,!*Picture)
sharePicture picture=:{pictOrigin,pictPen}
	= ({pictContext=0,pictToolbox=OSNewToolbox,pictOrigin=pictOrigin,pictPen=pictPen},picture)

defaultPen :: Pen
defaultPen
	= {	penSize			= 1
	  ,	penForeColour	= Black
	  ,	penBackColour	= White
	  ,	penPos			= zero
	  ,	penFont			= defaultFont
	  }
where
	(defaultFont,_)		= OSdefaultfont OSNewToolbox

dialogPen :: Pen
dialogPen
	= {	penSize			= 1
	  ,	penForeColour	= Black
	  ,	penBackColour	= White
	  ,	penPos			= zero
	  ,	penFont			= dialogFont
	  }
where
	(dialogFont,_)		= OSdialogfont OSNewToolbox


/*	Picture interface functions.
*/
apppicttoolbox :: !(IdFun *OSToolbox) !*Picture -> *Picture
apppicttoolbox f picture=:{pictToolbox}
	= {picture & pictToolbox=f pictToolbox}

accpicttoolbox :: !(St *OSToolbox .x) !*Picture -> (!.x,!*Picture)
accpicttoolbox f picture=:{pictToolbox}
	# (x,tb)	= f pictToolbox
	= (x,{picture & pictToolbox=tb})


/*	Attribute functions.
*/
getpictattributes :: !Picture -> [PictureAttribute]
getpictattributes {pictPen={penSize,penPos,penForeColour,penFont}}
	= [	PicturePenSize   penSize
	  ,	PicturePenPos    penPos
	  ,	PicturePenColour penForeColour
	  ,	PicturePenFont   penFont
	  ]

setpictattributes :: ![PictureAttribute] !*Picture -> *Picture
setpictattributes atts picture
	= StateMap2 setattribute (removeDup` atts) picture
where
	setattribute :: !PictureAttribute *Picture -> *Picture
	setattribute (PicturePenSize	size)	picture = setpictpensize   size picture
	setattribute (PicturePenPos		pos)	picture = setpictpenpos    pos  picture
	setattribute (PicturePenColour	col)	picture = setpictpencolour col  picture
	setattribute (PicturePenFont	font)	picture = setpictpenfont   font picture
	
	eqAtt :: !PictureAttribute !PictureAttribute -> Bool
	eqAtt (PicturePenSize _)	att = case att of
										(PicturePenSize _)		-> True
										_						-> False
	eqAtt (PicturePenPos _)		att = case att of
										(PicturePenPos _)		-> True
										_						-> False
	eqAtt (PicturePenColour _)	att = case att of
										(PicturePenColour _)	-> True
										_						-> False
	eqAtt (PicturePenFont _)	att = case att of
										(PicturePenFont _)		-> True
										_						-> False
	
	removeDup` :: ![PictureAttribute] -> [PictureAttribute]
	removeDup` [att:atts]
		= [att:removeDup` (filter (\att`->not (eqAtt att att`)) atts)]
	removeDup` _
		= []


//	Access to Origin and Pen:
getpictorigin :: !*Picture -> (!Origin,!*Picture)
getpictorigin picture=:{pictOrigin}
	= (pictOrigin,picture)

getpictpen :: !*Picture -> (!Pen,!*Picture)
getpictpen picture=:{pictPen}
	= (pictPen,picture)


//	Change the pen position:
setpictpenpos :: !Point !*Picture -> *Picture
setpictpenpos newpos picture=:{pictToolbox,pictOrigin,pictPen,pictContext}
	| newpos==pictPen.penPos
	= picture
	# (context,tb)	= WinMovePenTo (PointToTuple (newpos-pictOrigin)) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos=newpos}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

accpictpenpos :: !*Picture -> (!Point,!*Picture)
accpictpenpos picture=:{pictPen={penPos}}
	= (penPos,picture)

getpictpenpos :: !Picture -> Point
getpictpenpos {pictPen={penPos}}
	= penPos

movepictpenpos :: !Vector !*Picture -> *Picture
movepictpenpos v picture=:{pictToolbox,pictPen,pictContext}
	# (context,tb)	= WinMovePen (VectorToTuple v) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos=addPointVector v pictPen.penPos}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

//	Change the pen size:
setpictpensize :: !Int !*Picture -> *Picture
setpictpensize w picture=:{pictToolbox,pictContext,pictPen}
	| w`==pictPen.penSize
	= picture
	# (context,tb)	= WinSetPenSize w` (pictContext,pictToolbox)
	  pen			= {pictPen & penSize=w`}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	w` = max 1 w

accpictpensize :: !*Picture -> (!Int,!*Picture)
accpictpensize picture=:{pictPen={penSize}}
	= (penSize,picture)

getpictpensize :: !Picture -> Int
getpictpensize {pictPen={penSize}}
	= penSize

//	Change the PenColour:
setpictpencolour :: !Colour !*Picture -> *Picture
setpictpencolour colour picture=:{pictToolbox,pictPen,pictContext}
	# (context,tb)	= WinSetPenColor (toRGBtriple colour) (pictContext,pictToolbox)
	  pen			= {pictPen & penForeColour=colour}
	= {picture & pictPen=pen,pictToolbox=tb,pictContext=context}

toRGBtriple :: !Colour -> (!Int,!Int,!Int)
toRGBtriple (RGB {r,g,b})	= (SetBetween r MinRGB MaxRGB,SetBetween g MinRGB MaxRGB,SetBetween b MinRGB MaxRGB)
toRGBtriple Black			= (MinRGB,MinRGB,MinRGB)
toRGBtriple DarkGrey		= ( MaxRGB>>2,    MaxRGB>>2,    MaxRGB>>2)
toRGBtriple Grey			= ( MaxRGB>>1,    MaxRGB>>1,    MaxRGB>>1)
toRGBtriple LightGrey		= ((MaxRGB>>2)*3,(MaxRGB>>2)*3,(MaxRGB>>2)*3)
toRGBtriple White			= (MaxRGB,MaxRGB,MaxRGB)
toRGBtriple Red				= (MaxRGB,MinRGB,MinRGB)
toRGBtriple Green			= (MinRGB,MaxRGB,MinRGB)
toRGBtriple Blue			= (MinRGB,MinRGB,MaxRGB)
toRGBtriple Cyan			= (MinRGB,MaxRGB,MaxRGB)
toRGBtriple Magenta			= (MaxRGB,MinRGB,MaxRGB)
toRGBtriple Yellow			= (MaxRGB,MaxRGB,MinRGB)

accpictpencolour :: !*Picture -> (!Colour,!*Picture)
accpictpencolour picture=:{pictPen={penForeColour}}
	=(penForeColour,picture)

getpictpencolour :: !Picture -> Colour
getpictpencolour {pictPen={penForeColour}}
	= penForeColour

//	Change the font attributes:
setpictpenfont :: !Font !*Picture -> *Picture
setpictpenfont font picture=:{pictToolbox,pictContext,pictPen}
	# (context,tb)	= WinSetFont (osfontname,osfontstyles,osfontsize) (pictContext,pictToolbox)
	  pen			= {pictPen & penFont=font}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	{osfontname,osfontstyles,osfontsize}	= OSfontgetimp font

accpictpenfont :: !*Picture -> (!Font,!*Picture)
accpictpenfont picture=:{pictPen={penFont}}
	= (penFont,picture)

getpictpenfont :: !Picture -> Font
getpictpenfont {pictPen={penFont}}
	= penFont

setpictpendefaultfont :: !*Picture -> *Picture
setpictpendefaultfont picture=:{pictToolbox,pictContext,pictPen}
	# (font,tb)		= OSdefaultfont pictToolbox
	  {osfontname,osfontstyles,osfontsize}
	  				= OSfontgetimp font
	# (context,tb)	= WinSetFont (osfontname,osfontstyles,osfontsize) (pictContext,tb)
	  pen			= {pictPen & penFont=font}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}


/*	Drawing mode setting functions.
*/
setpictxormode :: !*Picture -> *Picture
setpictxormode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpicthilitemode :: !*Picture -> *Picture
setpicthilitemode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpictnormalmode :: !*Picture -> *Picture
setpictnormalmode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeCopy (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}


/*	Point drawing operations.
	pictdrawpoint
		only draws a point at that position. The pen position is not changed.
*/
pictdrawpoint :: !Point !*Picture -> *Picture
pictdrawpoint pos=:{x,y} picture=:{pictPen={penSize},pictOrigin={x=ox,y=oy},pictToolbox,pictContext}
	| penSize==1
		# (context,tb)	= WinDrawPoint (x`,y`) (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
	// otherwise
		# (context,tb)	= WinFillRectangle (x`,y`,x`+penSize,y`+penSize) (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
where
	(x`,y`)	= (x-ox,y-oy)


/*	Line drawing operations.
	pictdrawlineto
		draws a line from the current pen position to the given pen position. 
		The new pen position is the endpoint of the line.	
	pictdrawline
		draws a line from the first point to the second point. The pen position
		is not changed.
*/
pictdrawlineto :: !Point !*Picture -> *Picture
pictdrawlineto pos picture=:{pictOrigin,pictToolbox,pictContext,pictPen}
	# (context,tb)	= WinLinePenTo (PointToTuple (pos-pictOrigin)) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos=pos}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

pictdrawline :: !Point !Point !*Picture -> *Picture
pictdrawline a b picture=:{pictOrigin,pictToolbox,pictContext}
	# (context,tb)	= WinDrawLine (PointToTuple (a-pictOrigin)) (PointToTuple (b-pictOrigin)) (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}


/*	Text drawing operations.
	pictdraw(char/string) draws a char/string at the current pen position. The new
		pen position is immediately after the drawn char/string.
*/
pictdrawchar :: !Char !*Picture -> *Picture
pictdrawchar char picture=:{pictContext,pictToolbox,pictPen,pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinDrawChar (toInt char) (pictContext,pictToolbox)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	  pen					= {pictPen & penPos={x=x+ox,y=y+oy}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}

pictdrawstring :: !String !*Picture -> *Picture
pictdrawstring string picture=:{pictContext,pictToolbox,pictPen,pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinDrawString string (pictContext,pictToolbox)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	  pen					= {pictPen & penPos={x=x+ox,y=y+oy}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}


/*	Oval drawing operations.
	pict(draw/fill)oval center oval 
		draws/fills an oval at center with horizontal and vertical radius. The new
		pen position is not changed.
*/
pictdrawoval :: !Point !Oval !*Picture -> *Picture
pictdrawoval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinDrawOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictfilloval :: !Point !Oval !*Picture -> *Picture
pictfilloval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinFillOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

ovalToRect :: !Point !Oval -> (!Int,!Int,!Int,!Int)
ovalToRect {x,y} {oval_rx,oval_ry}
	= (x-rx,y-ry,x+rx,y+ry)
where
	rx	= abs oval_rx
	ry	= abs oval_ry


/*	Curve drawing operations.
	pict(draw/fill)curve movePen point curve
		draws/fills a curve starting at point with a shape defined by curve. If movePen
		is True, then the new pen position is at the end of the curve, otherwise it does
		not change.
*/
pictdrawcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictdrawcurve movePen start=:{x,y} curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinDrawCurve wrect (PointToTuple wstart) (PointToTuple wend) (pictContext,pictToolbox)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
	= picture
	= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictfillcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictfillcurve movePen start curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinFillWedge wrect (PointToTuple wstart) (PointToTuple wend) (pictContext,pictToolbox)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
	= picture
	= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

getcurve_rect_begin_end :: !Point !Curve -> (!(!Int,!Int,!Int,!Int),!Point,!Point)
getcurve_rect_begin_end start=:{x,y} {curve_oval={oval_rx,oval_ry},curve_from,curve_to,curve_clockwise}
	| curve_clockwise
	= (rect,end,start)
	= (rect,start,end)
where
	rx`	= toReal (abs oval_rx)
	ry`	= toReal (abs oval_ry)
	cx	= x -(toInt ((cos curve_from)*rx`))
	cy	= y +(toInt ((sin curve_from)*ry`))
	ex	= cx+(toInt ((cos curve_to  )*rx`))
	ey	= cy-(toInt ((sin curve_to  )*ry`))
	end	= {x=ex,y=ey}
	rect= (cx-oval_rx,cy-oval_ry,cx+oval_rx,cy+oval_ry)


/*	Rect drawing operations.
	pict(draw/fill)rect rect
		draws/fills a rect. The pen position is not changed.
*/
pictdrawrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictdrawrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy}}
	# rect			= (left-ox,top-oy,right-ox,bottom-oy)
	# (context,tb)	= WinDrawRectangle rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

pictfillrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictfillrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy}}
	# rect			= (left-ox,top-oy,right-ox,bottom-oy)
	# (context,tb)	= WinFillRectangle rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}


/*	Polygon drawing operations.
	pict(draw/fill)polygon point polygon
		draws/fills a polygon starting at point. The pen position is not changed.
*/
pictdrawpolygon :: !Point !Polygon !*Picture -> *Picture
pictdrawpolygon start {polygon_shape} picture=:{pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinDrawPolygon (pictContext,tb)
	# tb			= WinEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

pictfillpolygon :: !Point !Polygon !*Picture -> *Picture
pictfillpolygon start {polygon_shape} picture=:{pictPen={penSize},pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinSetPenSize 1 (pictContext,tb)
	# (context,tb)	= WinFillPolygon (context,tb)
	# (context,tb)	= WinDrawPolygon (context,tb)
	# (context,tb)	= WinSetPenSize penSize (context,tb)
	# tb			= WinEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

transferPolygon :: !Point ![Vector] !*OSToolbox -> *OSToolbox
transferPolygon start vs tb
	# tb	= WinStartPolygon (1 + length vs) tb
	# tb	= WinAddPolygonPoint wstart tb
	# tb	= transferShape wstart vs tb
	= tb
where
	wstart	= PointToTuple start
	
	transferShape :: !(!Int,!Int) ![Vector] !*OSToolbox -> *OSToolbox
	transferShape (x,y) [{vx,vy}:vs] tb
   		= transferShape newpos vs (WinAddPolygonPoint newpos tb)
	where
		newpos = (x+vx,y+vy)
	transferShape _ _ tb
		= tb

/*	Clipping operations.
	pictsetcliprgn sets the given clipping region.
*/
pictsetcliprgn :: !OSRgnHandle !*Picture -> (!OSRgnHandle,!*Picture)
pictsetcliprgn cliprgn picture=:{pictContext,pictToolbox}
	# (cliprgn,(context,tb))	= WinClipRgnPicture cliprgn (pictContext,pictToolbox)
	= (cliprgn,{picture & pictContext=context,pictToolbox=tb})
